
#include <stdio.h>
#include <math.h>

int main(void) {
    // 物理定数・初期条件
    const double a  = 9.8;     // 加速度 [m/s^2]
    const double x0 = 0.0;     // 初期位置 [m]
    const double v0 = 0.0;     // 初期速度 [m/s]

    // 時間設定
    const double t_start = 0.0;
    const double t_end   = 2.0;
    const double dt      = 0.1;   // 刻み [s]

    // 誤差判定用
    const double rel_tol = 1.0 / 1000.0;  // 相対誤差目標
    const double eps     = 1e-12;         // 0除算回避用

    // 状態変数
    double t = t_start;
    double x = x0;
    double v = v0;

    // ステップ数（端点を含む）
    int steps = (int)round((t_end - t_start) / dt);

    // ヘッダ
    printf("# t[s], x[m], x_exact[m], rel_error\n");

    // 初期点の出力
    {
        double x_exact = 0.5 * a * t * t;
        double denom   = fabs(x_exact) > eps ? fabs(x_exact) : 1.0; // t=0での0除算回避
        double rel_err = fabs(x - x_exact) / denom;
        printf("%.6f, %.12f, %.12f, %.6e\n", t, x, x_exact, rel_err);
    }

    // 時間発展（オイラー法）
    for (int n = 0; n < steps; ++n) {
        // 速度更新（v_{n+1} = v_n + a*dt）
        v += a * dt;

        // 位置更新（x_{n+1} = x_n + v_n*dt ではなく、更新後速度を使う場合は以下）
        // ここでは「速度→位置」の順に更新（オイラー法の一種、一次精度）
        x += v * dt;

        t = t_start + (n + 1) * dt;

        // 解析解と誤差
        double x_exact = 0.5 * a * t * t;
        double denom   = fabs(x_exact) > eps ? fabs(x_exact) : 1.0;
        double rel_err = fabs(x - x_exact) / denom;

        // 出力
        printf("%.6f, %.12f, %.12f, %.6e\n", t, x, x_exact, rel_err);

        // 相対誤差が許容を超えたら注意表示（終了はしない）
        if (rel_err > rel_tol) {
            fprintf(stderr,
                    "WARNING: t=%.3f s で相対誤差 %.3e が許容値 %.3e を超えています（dt=%.3f）。\n",
                    t, rel_err, rel_tol, dt);
        }
    }

    return 0;
}
