
#include <stdio.h>
#include <math.h>

int main(void) {
    // 物理定数・初期条件
    const double a  = 9.8;     // 加速度 [m/s^2]
    const double x0 = 0.0;     // 初期位置 [m]
    const double v0 = 0.0;     // 初期速度 [m/s]

    // 時間設定
    const double t_start = 0.0;
    const double t_end   = 2.0;
    const double dt      = 0.1;   // 刻み [s]

    // 誤差判定（相対誤差 1/1000 以下）
    const double rel_tol = 1.0 / 1000.0;
    const double eps     = 1e-12;  // 0除算回避用

    // ステップ数（端点を含むように初期点 + steps 回の更新）
    int steps = (int)round((t_end - t_start) / dt);

    // 状態変数
    double t = t_start;
    double x = x0;
    double v = v0;

    // 見出しなし：要求どおり t と x を画面に表示
    // 初期点の出力
    printf("%.6f %.12f\n", t, x);

    // 初期点の相対誤差チェック（参考：stderr に必要時のみ表示）
    {
        double x_exact = 0.5 * a * t * t;
        double denom   = fabs(x_exact) > eps ? fabs(x_exact) : 1.0;
        double rel_err = fabs(x - x_exact) / denom;
        if (rel_err > rel_tol) {
            fprintf(stderr,
                    "WARNING: t=%.3f s で相対誤差 %.3e が許容値 %.3e を超えています（dt=%.3f）。\n",
                    t, rel_err, rel_tol, dt);
        }
    }

    // 時間発展（古典的 4 次ルンゲクッタ法）
    for (int n = 0; n < steps; ++n) {
        // k1
        double k1x = v;
        double k1v = a;

        // k2（t + dt/2）
        double x2 = x + 0.5 * dt * k1x;
        double v2 = v + 0.5 * dt * k1v;
        double k2x = v2;
        double k2v = a; // a は定数

        // k3（t + dt/2）
        double x3 = x + 0.5 * dt * k2x;
        double v3 = v + 0.5 * dt * k2v;
        double k3x = v3;
        double k3v = a; // a は定数

        // k4（t + dt）
        double x4 = x + dt * k3x;
        double v4 = v + dt * k3v;
        double k4x = v4;
        double k4v = a; // a は定数

        // 更新
        x += (dt / 6.0) * (k1x + 2.0 * k2x + 2.0 * k3x + k4x);
        v += (dt / 6.0) * (k1v + 2.0 * k2v + 2.0 * k3v + k4v);

        t = t_start + (n + 1) * dt;

        // 出力（t と x のみ）
        printf("%.6f %.12f\n", t, x);

        // 解析解との相対誤差チェック（stderr に必要時のみ表示）
        double x_exact = 0.5 * a * t * t;
        double denom   = fabs(x_exact) > eps ? fabs(x_exact) : 1.0;
        double rel_err = fabs(x - x_exact) / denom;

        if (rel_err > rel_tol) {
            fprintf(stderr,
                    "WARNING: t=%.3f s で相対誤差 %.3e が許容値 %.3e を超えています（dt=%.3f）。\n",
                    t, rel_err, rel_tol, dt);
        }
    }

    return 0;
}
