#include <stdio.h>
#include <math.h>

// 物理定数と時間設定
#define G_ACCELERATION 9.8    // 加速度 a [m/s^2]
#define T_START 0.0           // 開始時間 t_start [s]
#define T_END 2.0             // 終了時間 t_end [s]
#define DISPLAY_DT 0.1        // 表示する時間刻み幅 display_dt [s]

// 相対誤差1/1000を達成するための計算刻み幅
#define CALC_DT 0.0001
// 表示刻み幅ごとに実行する計算回数 (0.1 / 0.0001 = 1000)
#define N_CALC_STEPS ((int)((DISPLAY_DT / CALC_DT) + 0.5)) 

int main(void) {
    // 初期条件
    double t = T_START;     // 時間 t [s]
    double x = 0.0;         // 位置 x [m] (初期位置 x0 = 0.0)
    double v = 0.0;         // 速度 v [m/s] (初期速度 v0 = 0.0)
    
    // 表示制御用カウンタ。0.1s経過したかどうかを判定するために使用
    int step_counter = 0;

    printf("--- 運動方程式のオイラー法による数値解 ---\n");
    printf("計算刻み幅 CALC_DT = %f [s] を使用し、精度を確保します。\n", CALC_DT);
    printf("------------------------------------------\n");
    printf("   t [s]   |   x [m] (数値解)\n");
    printf("-----------|------------------\n");

    // t_end (2.0秒)までシミュレーションを継続
    // ループは CALC_DT 刻みで実行
    while (t <= T_END + 1e-9) {
        
        // **表示**: step_counterがN_CALC_STEPSの倍数のとき、またはt=0のときに表示
        if (step_counter % N_CALC_STEPS == 0) {
            printf("  %8.2f |  %15.8f\n", t, x);
        }

        // --- オイラー法の適用 ---
        
        // 速度を更新 (v_{i+1} = v_i + a * dt)
        v = v + G_ACCELERATION * CALC_DT;
        
        // 位置を更新 (x_{i+1} = x_i + v_{i+1} * dt)
        x = x + v * CALC_DT; 

        // 時間を進行し、カウンタを更新
        t = t + CALC_DT;
        step_counter++;
    }

    // 最終的な相対誤差の情報を出力に追加
    double x_analytic = 0.5 * G_ACCELERATION * T_END * T_END;
    double relative_error = fabs((x_analytic - x) / x_analytic);
    
    printf("------------------------------------------\n");
    printf("相対誤差 (t=2.0s): %.5f\n", relative_error);
    
    return 0;
}