#include <stdio.h>
#include <math.h>

// 物理定数
#define G_ACCELERATION 9.8    // 加速度 a [m/s^2]
#define T_START 0.0           // 開始時間 t_start [s]
#define T_END 2.0             // 終了時間 t_end [s]
#define DISPLAY_DT 0.1        // 表示する時間刻み幅 display_dt [s]

// 計算刻み幅 (Leapfrog法は精度が高いため、オイラー法よりも少し大きめの刻み幅でも同程度の精度が出る場合がありますが、
// 安全を見て前回と同じ CALC_DT = 0.0001 [s]を採用します)
#define CALC_DT 0.0001
// 計算刻み数 (0.1 / 0.0001 = 1000)
#define N_CALC_STEPS ((int)((DISPLAY_DT / CALC_DT) + 0.5)) 

int main(void) {
    // 初期条件
    double t = T_START;     // 時間 t [s]
    double x = 0.0;         // 位置 x [m] (初期位置 x0 = 0.0)
    double v_full = 0.0;    // t=i の時刻でのフルステップ速度 v_i (出力用)
    
    // かえる跳び法の初期化ステップ:
    // v_halfは v_{i+1/2} を表す。最初の v_{-1/2} を計算する。
    // v_{-1/2} = v_0 - a * (DT / 2)
    double v_half = v_full - G_ACCELERATION * (CALC_DT / 2.0); 

    printf("--- 運動方程式の かえる跳び法 (Leapfrog) による数値解 ---\n");
    printf("加速度 a = %.1f [m/s^2]\n", G_ACCELERATION);
    printf("初期位置 x0 = %.1f [m], 初期速度 v0 = %.1f [m/s]\n", 0.0, 0.0);
    printf("計算刻み幅 CALC_DT = %f [s]\n", CALC_DT);
    printf("------------------------------------------\n");
    printf("   t [s]   |   x [m] (数値解)\n");
    printf("-----------|------------------\n");

    // t_end (2.0秒)までシミュレーションを継続
    // 浮動小数点数の比較誤差を考慮し、微小な値を加算
    while (t <= T_END + 1e-9) {
        
        // 0.1秒ごと（または初期時刻）に t と x を表示
        printf("  %8.2f |  %15.8f\n", t, x);

        // 表示用の0.1秒区間を、計算刻み幅CALC_DTでN_CALC_STEPS回繰り返す
        for (int i = 0; i < N_CALC_STEPS; i++) {
            
            // 1. 速度の半ステップ更新 (v_{i-1/2} -> v_{i+1/2})
            // 加速度 a は定数 G_ACCELERATION を使用
            v_half = v_half + G_ACCELERATION * CALC_DT;
            
            // 2. 位置のフルステップ更新 (x_i -> x_{i+1})
            x = x + v_half * CALC_DT;

            // 時間を進行
            t = t + CALC_DT;
            
            // T_ENDを超えたら、計算ループを終了
            if (t > T_END + 1e-9) break;
        }
    }

    // 最後に解析解との比較（参考情報）
    // 解析解: x = 1/2 * a * t^2
    double x_analytic = 0.5 * G_ACCELERATION * T_END * T_END;
    double relative_error = fabs((x_analytic - x) / x_analytic);
    
    printf("------------------------------------------\n");
    printf("t = %.1f [s] における解析解 x_analytic = %.8f [m]\n", T_END, x_analytic);
    printf("t = %.1f [s] における数値解 x_leapfrog = %.8f [m]\n", T_END, x);
    printf("t = %.1f [s] における相対誤差 (解析解との比較) = %.10f\n", T_END, relative_error);
    
    // 相対誤差1/1000 (0.001) のチェック
    if (relative_error < 0.001) {
        printf("相対誤差は1/1000 (0.001) 未満を満たしています。\n");
    } else {
        printf("相対誤差が1/1000 (0.001) を超えています。\n");
    }

    return 0;
}