#include <stdio.h>
#include <math.h>

// 物理定数
#define G_ACCELERATION 9.8    // 加速度 a [m/s^2]
#define T_START 0.0           // 開始時間 t_start [s]
#define T_END 2.0             // 終了時間 t_end [s]
#define CALC_DT 0.1           // 計算刻み幅 calc_dt [s] (表示刻み幅と同じ)

// 4次のルンゲ・クッタ法を適用する関数
void runge_kutta_step(double *x, double *v, double dt, double a) {
    // k_v: 速度の傾き dv/dt = a (定数)
    // k_x: 位置の傾き dx/dt = v 

    // k1: t_i における評価
    double k1v = a;
    double k1x = *v;

    // k2: t_i + dt/2 における評価
    double v_k2_est = *v + k1v * (dt / 2.0);
    double k2v = a;
    double k2x = v_k2_est;

    // k3: t_i + dt/2 における評価
    double v_k3_est = *v + k2v * (dt / 2.0);
    double k3v = a;
    double k3x = v_k3_est;

    // k4: t_i + dt における評価
    double v_k4_est = *v + k3v * dt;
    double k4v = a;
    double k4x = v_k4_est;

    // 位置 x の更新
    // x_{i+1} = x_i + (1/6) * (k1x + 2*k2x + 2*k3x + k4x) * dt
    *x = *x + (k1x + 2.0 * k2x + 2.0 * k3x + k4x) * (dt / 6.0);
    
    // 速度 v の更新
    // v_{i+1} = v_i + (1/6) * (k1v + 2*k2v + 2*k3v + k4v) * dt
    // (今回は k1v=k2v=k3v=k4v=a なので、v_{i+1} = v_i + a * dt と完全に一致する)
    *v = *v + (k1v + 2.0 * k2v + 2.0 * k3v + k4v) * (dt / 6.0);
}

int main(void) {
    // 初期条件
    double t = T_START;     // 時間 t [s]
    double x = 0.0;         // 位置 x [m] (初期位置 x0 = 0.0)
    double v = 0.0;         // 速度 v [m/s] (初期速度 v0 = 0.0)

    printf("--- 運動方程式の 4次のルンゲ・クッタ法 (RK4) による数値解 ---\n");
    printf("加速度 a = %.1f [m/s^2]\n", G_ACCELERATION);
    printf("初期位置 x0 = %.1f [m], 初期速度 v0 = %.1f [m/s]\n", 0.0, 0.0);
    printf("計算刻み幅 CALC_DT = %.1f [s]\n", CALC_DT);
    printf("------------------------------------------\n");
    printf("   t [s]   |   x [m] (数値解)\n");
    printf("-----------|------------------\n");

    // t_end (2.0秒)までシミュレーションを継続
    // 浮動小数点数の比較誤差を考慮し、微小な値を加算
    while (t <= T_END + 1e-9) {
        
        // t と x を表示
        printf("  %8.2f |  %15.8f\n", t, x);

        // RK4ステップを適用し、x, vを更新
        runge_kutta_step(&x, &v, CALC_DT, G_ACCELERATION);

        // 時間を進行
        t = t + CALC_DT;
    }

    // 最後に解析解との比較（参考情報）
    // 解析解: x = 1/2 * a * t^2
    double x_analytic = 0.5 * G_ACCELERATION * T_END * T_END;
    double relative_error = fabs((x_analytic - x) / x_analytic);
    
    printf("------------------------------------------\n");
    printf("t = %.1f [s] における解析解 x_analytic = %.8f [m]\n", T_END, x_analytic);
    printf("t = %.1f [s] における数値解 x_rk4 = %.8f [m]\n", T_END, x);
    printf("t = %.1f [s] における相対誤差 (解析解との比較) = %.12f\n", T_END, relative_error);
    
    // 相対誤差1/1000 (0.001) のチェック
    if (relative_error < 0.001) {
        printf("相対誤差は1/1000 (0.001) 未満を満たしています。\n");
    } else {
        printf("相対誤差が1/1000 (0.001) を超えています。\n");
    }

    return 0;
}